/**
 * script.js
 * Maneja:
 *  - Búsqueda dinámica (input) en tiempo real.
 *  - Paginación con botones Anterior/Siguiente y rango reducido de páginas.
 *  - Edición inline (blur).
 *  - Creación de nuevo producto (subida de imagen).
 *  - Eliminación de producto.
 */

let currentPage = 1;       // Página actual
let currentOrder = 'ASC';  // Orden por defecto
let currentSearch = '';    // Búsqueda actual
const limit = 5;           // Registros por página

document.addEventListener('DOMContentLoaded', function() {
  
  const searchInput = document.getElementById('searchInput');
  const orderSelect = document.getElementById('orderSelect');
  const btnBuscar   = document.getElementById('btnBuscar');

  // Búsqueda dinámica: al escribir, se dispara la carga de productos
  searchInput.addEventListener('input', () => {
    currentSearch = searchInput.value.trim();
    currentPage   = 1;
    loadProducts();
  });

  // Cambiar orden
  orderSelect.addEventListener('change', () => {
    currentOrder = orderSelect.value;
    currentPage  = 1;
    loadProducts();
  });

  // (Opcional) Si tuvieras un botón para buscar:
  if (btnBuscar) {
    btnBuscar.addEventListener('click', () => {
      currentSearch = searchInput.value.trim();
      currentOrder  = orderSelect.value;
      currentPage   = 1; // resetear a página 1
      loadProducts();
    });
  }

  // Carga inicial de productos
  loadProducts();

  // Manejo de submit del formulario de nuevo producto
  const formNuevo = document.getElementById('formNuevoProducto');
  formNuevo.addEventListener('submit', function(e){
    e.preventDefault();
    createProduct(formNuevo);
  });
});

/**
 * Función para cargar productos de la página actual.
 */
function loadProducts() {
  const bodyTable = document.getElementById('productosBody');
  const paginationUl = document.getElementById('paginationUl');

  // Limpiar tabla
  bodyTable.innerHTML = '<tr><td colspan="23">Cargando...</td></tr>';
  // Limpiar paginación
  paginationUl.innerHTML = '';

  let formData = new FormData();
  formData.append('page', currentPage);
  formData.append('limit', limit);
  formData.append('search', currentSearch);
  formData.append('order', currentOrder);

  fetch('productos_crud.php?action=list', {
    method: 'POST',
    body: formData
  })
  .then(response => response.json())
  .then(resp => {
    const productos = resp.data;
    const totalPages = resp.totalPages;

    if (productos.length === 0) {
      bodyTable.innerHTML = '<tr><td colspan="23" class="text-center">No hay resultados</td></tr>';
      return;
    }

    let rows = '';
    productos.forEach(prod => {
      const imagenTag = prod.imagen_producto 
                          ? `<img src="${prod.imagen_producto}" class="hover-image" alt="Imagen">`
                          : '';

      rows += `
        <tr data-id="${prod.id}">
          <td contenteditable="true" data-field="CodigoInterno">${prod.CodigoInterno ?? ''}</td>
          <td contenteditable="true" data-field="CodigoBarra">${prod.CodigoBarra ?? ''}</td>
          <td contenteditable="true" data-field="NombreComercial">${prod.NombreComercial ?? ''}</td>
          <td contenteditable="true" data-field="NombreGenerico">${prod.NombreGenerico ?? ''}</td>
          <td contenteditable="true" data-field="id_categoria">${prod.id_categoria ?? ''}</td>
          <td contenteditable="true" data-field="FuncionMedica">${prod.FuncionMedica ?? ''}</td>
          <td contenteditable="true" data-field="laboratorio">${prod.laboratorio ?? ''}</td>
          <td contenteditable="true" data-field="id_lugarCompra">${prod.id_lugarCompra ?? ''}</td>
          <td contenteditable="true" data-field="CostoMayor">${prod.CostoMayor ?? ''}</td>
          <td contenteditable="true" data-field="CostoBlister">${prod.CostoBlister ?? ''}</td>
          <td contenteditable="true" data-field="CostoUnidad">${prod.CostoUnidad ?? ''}</td>
          <td contenteditable="true" data-field="CostoMixto">${prod.CostoMixto ?? ''}</td>
          <td contenteditable="true" data-field="PrecioVentaUnidad">${prod.PrecioVentaUnidad ?? ''}</td>
          <td contenteditable="true" data-field="PrecioVentaBlister">${prod.PrecioVentaBlister ?? ''}</td>
          <td contenteditable="true" data-field="PrecioVentaMayor">${prod.PrecioVentaMayor ?? ''}</td>
          <td contenteditable="true" data-field="PrecioVentaMixto">${prod.PrecioVentaMixto ?? ''}</td>
          <td contenteditable="true" data-field="FechaIngreso">${prod.FechaIngreso ?? ''}</td>
          <td contenteditable="true" data-field="FechaVencimiento">${prod.FechaVencimiento ?? ''}</td>
          <td contenteditable="true" data-field="id_presentacion">${prod.id_presentacion ?? ''}</td>
          <td contenteditable="true" data-field="estado">${prod.estado ?? ''}</td>
          <td contenteditable="true" data-field="factor_conversion">${prod.factor_conversion ?? ''}</td>
          <td class="product-hover">
            ${imagenTag}
            ${prod.imagen_producto ? 'Ver Imagen' : 'Sin imagen'}
          </td>
          <td>
            <button 
              class="btn btn-danger btn-sm"
              onclick="deleteProduct(${prod.id})"
            >
              Eliminar
            </button>
          </td>
        </tr>
      `;
    });

    bodyTable.innerHTML = rows;

    // Generar paginación con "Anterior", rango de páginas y "Siguiente"
    createPagination(totalPages);

    // Agregar event listeners para edición inline (blur)
    document.querySelectorAll('#productosBody td[contenteditable="true"]').forEach(cell => {
      cell.addEventListener('blur', function(e){
        const newValue = e.target.innerText.trim();
        const field = e.target.getAttribute('data-field');
        const tr = e.target.closest('tr');
        const productId = tr.getAttribute('data-id');
        updateField(productId, field, newValue);
      });
    });

  })
  .catch(err => {
    console.error(err);
    bodyTable.innerHTML = '<tr><td colspan="23" class="text-center text-danger">Error al cargar</td></tr>';
  });
}

/**
 * Crea la estructura de paginación con botones anterior, siguiente,
 * y un rango de páginas alrededor de la página actual.
 * @param {number} totalPages 
 */
function createPagination(totalPages) {
  const paginationUl = document.getElementById('paginationUl');
  paginationUl.innerHTML = '';

  // Botón "Anterior"
  let prevLi = document.createElement('li');
  prevLi.classList.add('page-item');
  if (currentPage === 1) {
    prevLi.classList.add('disabled');
  }
  let prevLink = document.createElement('a');
  prevLink.classList.add('page-link');
  prevLink.textContent = 'Anterior';
  prevLink.href = '#';
  prevLink.addEventListener('click', (e) => {
    e.preventDefault();
    if (currentPage > 1) {
      currentPage--;
      loadProducts();
    }
  });
  prevLi.appendChild(prevLink);
  paginationUl.appendChild(prevLi);

  // Páginas intermedias
  const maxRange = 2;
  let startPage = currentPage - maxRange;
  let endPage   = currentPage + maxRange;

  if (startPage < 1) startPage = 1;
  if (endPage > totalPages) endPage = totalPages;

  for (let i = startPage; i <= endPage; i++) {
    let li = document.createElement('li');
    li.classList.add('page-item');
    if (i === currentPage) {
      li.classList.add('active');
    }
    let a = document.createElement('a');
    a.classList.add('page-link');
    a.textContent = i;
    a.href = "#";
    a.addEventListener('click', function(e){
      e.preventDefault();
      currentPage = i;
      loadProducts();
    });
    li.appendChild(a);
    paginationUl.appendChild(li);
  }

  // Botón "Siguiente"
  let nextLi = document.createElement('li');
  nextLi.classList.add('page-item');
  if (currentPage === totalPages || totalPages === 0) {
    nextLi.classList.add('disabled');
  }
  let nextLink = document.createElement('a');
  nextLink.classList.add('page-link');
  nextLink.textContent = 'Siguiente';
  nextLink.href = '#';
  nextLink.addEventListener('click', (e) => {
    e.preventDefault();
    if (currentPage < totalPages) {
      currentPage++;
      loadProducts();
    }
  });
  nextLi.appendChild(nextLink);
  paginationUl.appendChild(nextLi);
}

/**
 * Función para actualizar un campo del producto (edición inline).
 */
function updateField(id, field, value) {
  let formData = new FormData();
  formData.append('id', id);
  formData.append('field', field);
  formData.append('value', value);

  fetch('productos_crud.php?action=update', {
    method: 'POST',
    body: formData
  })
  .then(response => response.json())
  .then(resp => {
    if(resp.status === 'success') {
      console.log('Actualizado correctamente');
    } else {
      console.warn(resp.message);
    }
  })
  .catch(err => {
    console.error('Error actualizando:', err);
  });
}

/**
 * Función para crear un nuevo producto (subida de imagen).
 */
function createProduct(form) {
  let formData = new FormData(form);

  fetch('productos_crud.php?action=create', {
    method: 'POST',
    body: formData
  })
  .then(response => response.json())
  .then(resp => {
    if(resp.status === 'success'){
      alert(resp.message);
      // Cerrar modal
      let modalEl = document.getElementById('modalNuevoProducto');
      let modal = bootstrap.Modal.getInstance(modalEl);
      modal.hide();

      // Resetear formulario
      form.reset();

      // Recargar la tabla
      loadProducts();
    } else {
      alert('Error: ' + resp.message);
    }
  })
  .catch(err => {
    console.error(err);
    alert('Ocurrió un error al crear el producto.');
  });
}

/**
 * Función para eliminar un producto.
 */
function deleteProduct(id) {
  if(!confirm("¿Estás seguro de eliminar este producto?")) {
    return;
  }

  let formData = new FormData();
  formData.append('id', id);

  fetch('productos_crud.php?action=delete', {
    method: 'POST',
    body: formData
  })
  .then(res => res.json())
  .then(resp => {
    if(resp.status === 'success') {
      alert(resp.message);
      // Recargar la lista
      loadProducts();
    } else {
      alert('Error: ' + resp.message);
    }
  })
  .catch(err => {
    console.error(err);
    alert('Error al eliminar el producto.');
  });
}
