<?php
/**
 * productos_crud.php
 * CRUD sobre la tabla 'productos' mediante AJAX.
 */
require_once 'db.php';

$action = isset($_GET['action']) ? $_GET['action'] : '';

switch ($action) {

    // ------------------------------------------------------------------
    // LISTAR (con paginación, búsqueda y ordenamiento)
    // ------------------------------------------------------------------
    case 'list':
        // Parámetros de paginación, búsqueda, orden
        $page   = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $limit  = isset($_POST['limit']) ? intval($_POST['limit']) : 5;
        $search = isset($_POST['search']) ? trim($_POST['search']) : '';
        $order  = isset($_POST['order']) ? $_POST['order'] : 'ASC';
        $order  = strtoupper($order) === 'DESC' ? 'DESC' : 'ASC';
        $offset = ($page - 1) * $limit;

        // ------------------------------------------------------------------
        // BÚSQUEDA CON COMBINACIONES DE PALABRAS
        // ------------------------------------------------------------------
        // Dividimos el input de búsqueda en "tokens" (palabras),
        // y para cada palabra generamos un OR en varios campos,
        // y luego un AND para todas las palabras.

        $tokens = array_filter(explode(' ', $search)); // dividir por espacios
        $conditions = [];
        $params = [];
        $i = 0;

        // Campos a buscar
        $fields = [
            'CodigoInterno',
            'NombreComercial',
            'NombreGenerico',
            'laboratorio',
            'FuncionMedica',
            'FechaIngreso',
            'FechaVencimiento'
        ];

        foreach ($tokens as $tok) {
            $tok = trim($tok);
            if ($tok !== '') {
                // Construimos algo como:
                // (CodigoInterno LIKE :token0 OR NombreComercial LIKE :token0 OR ...)
                $subConditions = [];
                foreach ($fields as $f) {
                    $paramName = ":token{$i}";
                    // Ejemplo: "CodigoInterno LIKE :token0"
                    $subConditions[] = "$f LIKE $paramName";
                }
                // Unir cada subcondición con OR
                $conditions[] = '(' . implode(' OR ', $subConditions) . ')';

                // Guardar el valor a buscar con comodines
                $params[":token{$i}"] = "%{$tok}%";
                $i++;
            }
        }

        $where = '';
        if (count($conditions) > 0) {
            // Unir con AND cada grupo de OR
            $where = 'WHERE ' . implode(' AND ', $conditions);
        }

        // Contar total de registros
        $countSql = "SELECT COUNT(*) as total FROM productos $where";
        $countStmt = $pdo->prepare($countSql);
        // Vincular parámetros de búsqueda
        foreach ($params as $pKey => $pVal) {
            $countStmt->bindValue($pKey, $pVal, PDO::PARAM_STR);
        }
        $countStmt->execute();
        $totalRow = $countStmt->fetch(PDO::FETCH_ASSOC);
        $totalRegistros = $totalRow['total'];

        // Obtener registros paginados
        $sql = "SELECT *
                FROM productos
                $where
                ORDER BY CodigoInterno $order
                LIMIT :offset, :limit";
        $stmt = $pdo->prepare($sql);

        // Vincular parámetros de búsqueda
        foreach ($params as $pKey => $pVal) {
            $stmt->bindValue($pKey, $pVal, PDO::PARAM_STR);
        }
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);

        $stmt->execute();
        $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Calcular número de páginas
        $totalPaginas = ($limit > 0) ? ceil($totalRegistros / $limit) : 1;

        // Respuesta JSON
        echo json_encode([
            'data'         => $productos,
            'totalRecords' => $totalRegistros,
            'totalPages'   => $totalPaginas
        ]);
    break;

    // ------------------------------------------------------------------
    // CREAR (subir imagen incluida)
    // ------------------------------------------------------------------
    case 'create':
        $CodigoInterno        = $_POST['CodigoInterno'] ?? '';
        $CodigoBarra          = $_POST['CodigoBarra'] ?? '';
        $NombreComercial      = $_POST['NombreComercial'] ?? '';
        $NombreGenerico       = $_POST['NombreGenerico'] ?? '';
        $id_categoria         = $_POST['id_categoria'] ?? null;
        $FuncionMedica        = $_POST['FuncionMedica'] ?? '';
        $laboratorio          = $_POST['laboratorio'] ?? '';
        $id_lugarCompra       = $_POST['id_lugarCompra'] ?? null;
        $CostoMayor           = $_POST['CostoMayor'] ?? null;
        $CostoBlister         = $_POST['CostoBlister'] ?? null;
        $CostoUnidad          = $_POST['CostoUnidad'] ?? null;
        $CostoMixto           = $_POST['CostoMixto'] ?? null;
        $PrecioVentaUnidad    = $_POST['PrecioVentaUnidad'] ?? null;
        $PrecioVentaBlister   = $_POST['PrecioVentaBlister'] ?? null;
        $PrecioVentaMayor     = $_POST['PrecioVentaMayor'] ?? null;
        $PrecioVentaMixto     = $_POST['PrecioVentaMixto'] ?? null;
        $FechaIngreso         = $_POST['FechaIngreso'] ?? null;
        $FechaVencimiento     = $_POST['FechaVencimiento'] ?? null;
        $id_presentacion      = $_POST['id_presentacion'] ?? null;
        $estado               = isset($_POST['estado']) ? $_POST['estado'] : 1;
        $factor_conversion    = $_POST['factor_conversion'] ?? 1;

        $imagen_producto = '';
        if (!empty($_FILES['imagen_producto']['name'])) {
            $uploadDir = 'uploads/';
            if(!is_dir($uploadDir)){
                mkdir($uploadDir, 0777, true);
            }
            $filename   = time() . '_' . basename($_FILES['imagen_producto']['name']);
            $targetFile = $uploadDir . $filename;
            if (move_uploaded_file($_FILES['imagen_producto']['tmp_name'], $targetFile)) {
                $imagen_producto = $targetFile;
            }
        }

        $sql = "INSERT INTO productos (
                    CodigoInterno, CodigoBarra, NombreComercial, NombreGenerico, id_categoria, 
                    FuncionMedica, laboratorio, id_lugarCompra, 
                    CostoMayor, CostoBlister, CostoUnidad, CostoMixto,
                    PrecioVentaUnidad, PrecioVentaBlister, PrecioVentaMayor, PrecioVentaMixto,
                    FechaIngreso, FechaVencimiento, imagen_producto, 
                    id_presentacion, estado, factor_conversion
                ) VALUES (
                    :CodigoInterno, :CodigoBarra, :NombreComercial, :NombreGenerico, :id_categoria,
                    :FuncionMedica, :laboratorio, :id_lugarCompra,
                    :CostoMayor, :CostoBlister, :CostoUnidad, :CostoMixto,
                    :PrecioVentaUnidad, :PrecioVentaBlister, :PrecioVentaMayor, :PrecioVentaMixto,
                    :FechaIngreso, :FechaVencimiento, :imagen_producto,
                    :id_presentacion, :estado, :factor_conversion
                )";

        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':CodigoInterno', $CodigoInterno);
        $stmt->bindValue(':CodigoBarra', $CodigoBarra);
        $stmt->bindValue(':NombreComercial', $NombreComercial);
        $stmt->bindValue(':NombreGenerico', $NombreGenerico);
        $stmt->bindValue(':id_categoria', $id_categoria);
        $stmt->bindValue(':FuncionMedica', $FuncionMedica);
        $stmt->bindValue(':laboratorio', $laboratorio);
        $stmt->bindValue(':id_lugarCompra', $id_lugarCompra);
        $stmt->bindValue(':CostoMayor', $CostoMayor);
        $stmt->bindValue(':CostoBlister', $CostoBlister);
        $stmt->bindValue(':CostoUnidad', $CostoUnidad);
        $stmt->bindValue(':CostoMixto', $CostoMixto);
        $stmt->bindValue(':PrecioVentaUnidad', $PrecioVentaUnidad);
        $stmt->bindValue(':PrecioVentaBlister', $PrecioVentaBlister);
        $stmt->bindValue(':PrecioVentaMayor', $PrecioVentaMayor);
        $stmt->bindValue(':PrecioVentaMixto', $PrecioVentaMixto);
        $stmt->bindValue(':FechaIngreso', $FechaIngreso);
        $stmt->bindValue(':FechaVencimiento', $FechaVencimiento);
        $stmt->bindValue(':imagen_producto', $imagen_producto);
        $stmt->bindValue(':id_presentacion', $id_presentacion);
        $stmt->bindValue(':estado', $estado, PDO::PARAM_INT);
        $stmt->bindValue(':factor_conversion', $factor_conversion, PDO::PARAM_INT);

        $stmt->execute();

        echo json_encode(["status" => "success", "message" => "Producto creado correctamente."]);
    break;

    // ------------------------------------------------------------------
    // ACTUALIZAR un campo (edición inline)
    // ------------------------------------------------------------------
    case 'update':
        $id    = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $field = $_POST['field'] ?? '';
        $value = $_POST['value'] ?? '';

        if ($field === 'id') {
            echo json_encode(["status" => "error", "message" => "No se puede editar el campo 'id'."]);
            exit;
        }

        $validFields = [
            "CodigoInterno", "CodigoBarra", "NombreComercial", "NombreGenerico", "id_categoria", 
            "FuncionMedica", "laboratorio", "id_lugarCompra", 
            "CostoMayor", "CostoBlister", "CostoUnidad", "CostoMixto",
            "PrecioVentaUnidad", "PrecioVentaBlister", "PrecioVentaMayor", "PrecioVentaMixto",
            "FechaIngreso", "FechaVencimiento", "imagen_producto", 
            "id_presentacion", "estado", "factor_conversion"
        ];

        if (!in_array($field, $validFields)) {
            echo json_encode(["status" => "error", "message" => "Campo inválido."]);
            exit;
        }

        $sql = "UPDATE productos SET $field = :value WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':value', $value);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();

        echo json_encode(["status" => "success", "message" => "Producto actualizado."]);
    break;

    // ------------------------------------------------------------------
    // ELIMINAR
    // ------------------------------------------------------------------
    case 'delete':
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        if ($id <= 0) {
            echo json_encode(["status" => "error", "message" => "ID de producto inválido."]);
            exit;
        }

        $sql = "DELETE FROM productos WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':id', $id, PDO::PARAM_INT);
        $stmt->execute();

        echo json_encode(["status" => "success", "message" => "Producto eliminado correctamente."]);
    break;

    // ------------------------------------------------------------------
    // DEFAULT (acción desconocida)
    // ------------------------------------------------------------------
    default:
        echo json_encode(["status" => "error", "message" => "Acción no válida."]);
    break;
}
