document.addEventListener('DOMContentLoaded', () => {

    const searchInput = document.getElementById('search-input');
    const labFilter = document.getElementById('lab-filter-input');
    const tableContainer = document.getElementById('product-table-container');
    const tableBody = document.getElementById('product-table-body');
    const paginationContainer = document.getElementById('pagination-container');

    let debounceTimer;

    // --- Función Debounce ---
    // Evita que se hagan 1000 llamadas al servidor mientras escribes.
    // Espera 300ms después de la última tecla antes de buscar.
    function debounce(func, delay = 300) {
        return function(...args) {
            clearTimeout(debounceTimer);
            debounceTimer = setTimeout(() => {
                func.apply(this, args);
            }, delay);
        };
    }

    // --- NUEVA Función de Búsqueda con AJAX ---
    async function fetchProducts(page = 1) {
        const searchTerm = searchInput.value;
        const labTerm = labFilter.value;
        
        // Muestra feedback visual
        tableContainer.style.opacity = '0.5';

        try {
            // Construye la URL para el API
            const params = new URLSearchParams({
                search: searchTerm,
                lab: labTerm,
                page: page
            });
            
            const response = await fetch(`api_search_products.php?${params}`);
            if (!response.ok) throw new Error('Error en la respuesta del servidor.');

            const data = await response.json();

            // Reemplaza el contenido de la tabla y la paginación
            tableBody.innerHTML = data.html_table;
            paginationContainer.innerHTML = data.html_pagination;

        } catch (error) {
            console.error('Error al buscar productos:', error);
            tableBody.innerHTML = '<tr><td colspan="5" class="px-6 py-12 text-center text-red-500">Error al cargar los productos.</td></tr>';
        } finally {
            tableContainer.style.opacity = '1';
        }
    }

    // --- Lógica de Guardado de Categoría (SIN CAMBIOS) ---
    tableBody.addEventListener('change', async (event) => {
        if (event.target.classList.contains('category-select')) {
            const select = event.target;
            const row = select.closest('tr');
            const indicatorCell = row.querySelector('.saving-indicator');
            const productId = select.dataset.productId;
            const categoryId = select.value;

            select.disabled = true;
            row.style.opacity = '0.7';
            indicatorCell.innerHTML = '<div class="loader"></div>';

            try {
                const response = await fetch('api_update_category.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json', 'Accept': 'application/json'},
                    body: JSON.stringify({
                        producto_id: productId,
                        categoria_id: categoryId
                    })
                });
                if (!response.ok) throw new Error('Error al guardar.');
                const result = await response.json();
                if (result.status === 'success') {
                    row.style.backgroundColor = '#ecfdf5';
                } else {
                    throw new Error(result.message);
                }
            } catch (error) {
                console.error('Error al guardar categoría:', error);
                row.style.backgroundColor = '#fef2f2';
            } finally {
                setTimeout(() => {
                    select.disabled = false;
                    row.style.opacity = '1';
                    row.style.backgroundColor = '';
                    indicatorCell.innerHTML = '';
                }, 700);
            }
        }
    });

    // --- NUEVOS Listeners para Búsqueda y Paginación ---

    // Llama a fetchProducts (reseteando a pág 1) cuando escribes
    const debouncedFetch = debounce(() => fetchProducts(1), 300);
    searchInput.addEventListener('keyup', debouncedFetch);
    labFilter.addEventListener('keyup', debouncedFetch);

    // Intercepta clics en la paginación para usar AJAX
    paginationContainer.addEventListener('click', (event) => {
        // Encuentra el enlace <a> más cercano en el que se hizo clic
        const link = event.target.closest('a.pagination-link');
        
        if (link) {
            event.preventDefault(); // Evita la recarga de la página
            
            // No hacer nada si el enlace está deshabilitado
            if (link.classList.contains('opacity-50')) {
                return;
            }

            // Obtiene la página de la URL del enlace
            const url = new URL(link.href);
            const page = url.searchParams.get('page') || 1;
            
            // Llama a fetchProducts con la nueva página
            fetchProducts(page);
        }
    });
});