<?php
/**
 * API: Delete Admin Order Detail
 * Elimina detalles de pedidos desde el panel de administrador
 * Soporta eliminación en vista consolidada y vista de sucursal única
 */

header('Access-Control-Allow-Origin: *');
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../config/database.php';

// Solo permitir POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método no permitido']);
    exit;
}

try {
    $pdo = getDBConnection();
    
    // Obtener datos
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    // Validar parámetros requeridos
    if (!isset($data['producto_id'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'ID de producto requerido']);
        exit;
    }
    
    $producto_id = intval($data['producto_id']);
    $sucursal_id = isset($data['sucursal_id']) ? intval($data['sucursal_id']) : 0;
    
    // Construir condiciones WHERE
    $whereConditions = ['dp.producto_id = :producto_id'];
    $params = [':producto_id' => $producto_id];
    
    // Si se especifica sucursal, filtrar por ella
    if ($sucursal_id > 0) {
        $whereConditions[] = 'p.sucursal_id = :sucursal_id';
        $params[':sucursal_id'] = $sucursal_id;
    }
    
    // Agregar filtros adicionales si se proporcionan (para mayor precisión)
    if (isset($data['laboratorio']) && !empty($data['laboratorio'])) {
        $whereConditions[] = 'prod.laboratorio LIKE :laboratorio';
        $params[':laboratorio'] = "%" . $data['laboratorio'] . "%";
    }
    
    if (isset($data['categoria_id']) && $data['categoria_id'] > 0) {
        $whereConditions[] = 'prod.id_categoria = :categoria_id';
        $params[':categoria_id'] = intval($data['categoria_id']);
    }
    
    if (isset($data['fecha_inicio']) && !empty($data['fecha_inicio'])) {
        $whereConditions[] = 'DATE(p.fecha_pedido) >= :fecha_inicio';
        $params[':fecha_inicio'] = $data['fecha_inicio'];
    }
    
    if (isset($data['fecha_fin']) && !empty($data['fecha_fin'])) {
        $whereConditions[] = 'DATE(p.fecha_pedido) <= :fecha_fin';
        $params[':fecha_fin'] = $data['fecha_fin'];
    }
    
    if (isset($data['es_urgente']) && $data['es_urgente']) {
        $whereConditions[] = 'dp.es_urgente = 1';
    }
    
    $whereClause = implode(' AND ', $whereConditions);
    
    // Primero, verificar que existan registros y que sean de pedidos PENDIENTES
    $sqlCheck = "SELECT dp.id, p.estado, s.nombre as sucursal_nombre, dp.nombre_comercial
                 FROM detalle_pedidos dp
                 INNER JOIN pedidos p ON dp.pedido_id = p.id
                 INNER JOIN sucursales s ON p.sucursal_id = s.id
                 INNER JOIN productos prod ON dp.producto_id = prod.id
                 WHERE {$whereClause}";
                 
    $stmtCheck = $pdo->prepare($sqlCheck);
    $stmtCheck->execute($params);
    $records = $stmtCheck->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($records)) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'No se encontraron registros para eliminar']);
        exit;
    }
    
    // Verificar que todos los registros sean de pedidos pendientes
    $nonPendingRecords = array_filter($records, function($record) {
        return $record['estado'] !== 'pendiente';
    });
    
    if (!empty($nonPendingRecords)) {
        http_response_code(403);
        echo json_encode([
            'success' => false, 
            'error' => 'No se puede eliminar: Algunos pedidos ya fueron procesados o finalizados'
        ]);
        exit;
    }
    
    // Obtener IDs de los detalles a eliminar
    $detailIds = array_column($records, 'id');
    
    // Proceder a eliminar
    $placeholders = implode(',', array_fill(0, count($detailIds), '?'));
    $sqlDelete = "DELETE FROM detalle_pedidos WHERE id IN ({$placeholders})";
    $stmtDelete = $pdo->prepare($sqlDelete);
    $stmtDelete->execute($detailIds);
    
    $deletedCount = $stmtDelete->rowCount();
    
    echo json_encode([
        'success' => true, 
        'message' => "Se eliminaron {$deletedCount} registro(s) correctamente",
        'deleted_count' => $deletedCount
    ]);
    
} catch (PDOException $e) {
    error_log("Error al eliminar pedido: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Error de base de datos']);
}
?>
