<?php
/**
 * API: Guardar Pedido
 * Guarda un pedido completo con sus detalles
 */

header('Access-Control-Allow-Origin: *');
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/branch.php';

try {
    $pdo = getDBConnection();
    
    // Obtener datos del POST
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    // Validar datos requeridos
    if (!isset($data['usuario_id']) || !isset($data['productos'])) {
        sendJSON(['success' => false, 'error' => 'Datos incompletos'], 400);
    }
    
    $sucursal_id = getCurrentBranchId();
    $usuario_id = intval($data['usuario_id']);
    $productos = $data['productos'];
    $observaciones = isset($data['observaciones']) ? sanitize($data['observaciones']) : null;
    
    // Validar que haya productos
    if (empty($productos) || !is_array($productos)) {
        sendJSON(['success' => false, 'error' => 'Debe incluir al menos un producto'], 400);
    }
    
    // Iniciar transacción
    $pdo->beginTransaction();
    
    try {
        // Insertar pedido principal
        $sqlPedido = "INSERT INTO pedidos (sucursal_id, usuario_id, observaciones) 
                      VALUES (:sucursal_id, :usuario_id, :observaciones)";
        $stmtPedido = $pdo->prepare($sqlPedido);
        $stmtPedido->execute([
            ':sucursal_id' => $sucursal_id,
            ':usuario_id' => $usuario_id,
            ':observaciones' => $observaciones
        ]);
        
        $pedido_id = $pdo->lastInsertId();
        
        // Insertar detalles del pedido
        $sqlDetalle = "INSERT INTO detalle_pedidos (
                        pedido_id, producto_id, codigo_interno, codigo_barra,
                        nombre_comercial, laboratorio, es_urgente,
                        existencia_actual, presentacion_actual,
                        cantidad_solicitada, presentacion_solicitada
                      ) VALUES (
                        :pedido_id, :producto_id, :codigo_interno, :codigo_barra,
                        :nombre_comercial, :laboratorio, :es_urgente,
                        :existencia_actual, :presentacion_actual,
                        :cantidad_solicitada, :presentacion_solicitada
                      )";
        
        $stmtDetalle = $pdo->prepare($sqlDetalle);
        
        // Preparar statement para verificar duplicados
        $sqlCheckDup = "SELECT dp.id 
                        FROM detalle_pedidos dp 
                        INNER JOIN pedidos p ON dp.pedido_id = p.id 
                        WHERE p.sucursal_id = :sucursal_id 
                        AND p.estado = 'pendiente' 
                        AND dp.producto_id = :producto_id";
        $stmtCheckDup = $pdo->prepare($sqlCheckDup);

        foreach ($productos as $producto) {
            // Verificar si ya existe
            $stmtCheckDup->execute([
                ':sucursal_id' => $sucursal_id,
                ':producto_id' => intval($producto['producto_id'])
            ]);
            
            if ($stmtCheckDup->fetch()) {
                throw new Exception("El producto '{$producto['nombre_comercial']}' ya está en un pedido pendiente. Actualice la página.");
            }

            $stmtDetalle->execute([
                ':pedido_id' => $pedido_id,
                ':producto_id' => intval($producto['producto_id']),
                ':codigo_interno' => isset($producto['codigo_interno']) ? sanitize($producto['codigo_interno']) : null,
                ':codigo_barra' => isset($producto['codigo_barra']) ? sanitize($producto['codigo_barra']) : null,
                ':nombre_comercial' => sanitize($producto['nombre_comercial']),
                ':laboratorio' => isset($producto['laboratorio']) ? sanitize($producto['laboratorio']) : null,
                ':es_urgente' => isset($producto['es_urgente']) ? intval($producto['es_urgente']) : 0,
                ':existencia_actual' => intval($producto['existencia_actual']),
                ':presentacion_actual' => sanitize($producto['presentacion_actual']),
                ':cantidad_solicitada' => intval($producto['cantidad_solicitada']),
                ':presentacion_solicitada' => sanitize($producto['presentacion_solicitada'])
            ]);
        }
        
        // Confirmar transacción
        $pdo->commit();
        
        sendJSON([
            'success' => true,
            'message' => 'Pedido creado exitosamente',
            'pedido_id' => $pedido_id
        ]);
        
    } catch (Exception $e) {
        // Revertir transacción en caso de error
        $pdo->rollBack();
        throw $e;
    }
    
} catch (PDOException $e) {
    error_log("Error al guardar pedido: " . $e->getMessage());
    sendJSON(['success' => false, 'error' => 'Error al guardar el pedido'], 500);
} catch (Exception $e) {
    error_log("Error general: " . $e->getMessage());
    sendJSON(['success' => false, 'error' => 'Error al procesar la solicitud'], 500);
}
