<?php
/**
 * API: Actualizar Detalle de Pedido
 * Actualiza la cantidad, urgencia, etc. de un producto ya pedido
 */

header('Access-Control-Allow-Origin: *');
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/branch.php';

try {
    $pdo = getDBConnection();
    
    // Obtener datos del POST
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    // Validar datos requeridos
    if (!isset($data['detalle_id']) || !isset($data['cantidad_solicitada'])) {
        sendJSON(['success' => false, 'error' => 'Datos incompletos'], 400);
    }
    
    $detalle_id = intval($data['detalle_id']);
    $cantidad = intval($data['cantidad_solicitada']);
    $usuario_id = isset($data['usuario_id']) ? intval($data['usuario_id']) : null;
    $es_urgente = isset($data['es_urgente']) ? intval($data['es_urgente']) : 0;
    $existencia = isset($data['existencia_actual']) ? intval($data['existencia_actual']) : 0;
    $pres_actual = isset($data['presentacion_actual']) ? sanitize($data['presentacion_actual']) : 'UNIDAD';
    $pres_solicitada = isset($data['presentacion_solicitada']) ? sanitize($data['presentacion_solicitada']) : 'UNIDAD';
    
    // Verificar que el detalle pertenezca a un pedido de la sucursal actual
    // y que el pedido esté en estado 'pendiente'
    $sucursal_id = getCurrentBranchId();
    
    $sqlCheck = "SELECT dp.id, p.id as pedido_id 
                 FROM detalle_pedidos dp
                 INNER JOIN pedidos p ON dp.pedido_id = p.id
                 WHERE dp.id = :detalle_id 
                 AND p.sucursal_id = :sucursal_id 
                 AND p.estado = 'pendiente'";
                 
    $stmtCheck = $pdo->prepare($sqlCheck);
    $stmtCheck->execute([
        ':detalle_id' => $detalle_id,
        ':sucursal_id' => $sucursal_id
    ]);
    
    $detalle = $stmtCheck->fetch(PDO::FETCH_ASSOC);
    
    if (!$detalle) {
        sendJSON(['success' => false, 'error' => 'Pedido no encontrado o no editable'], 404);
    }
    
    // Iniciar transacción
    $pdo->beginTransaction();
    
    try {
        // Actualizar detalle
        $sqlUpdate = "UPDATE detalle_pedidos SET 
                        cantidad_solicitada = :cantidad,
                        es_urgente = :es_urgente,
                        existencia_actual = :existencia,
                        presentacion_actual = :pres_actual,
                        presentacion_solicitada = :pres_solicitada
                      WHERE id = :detalle_id";
                      
        $stmtUpdate = $pdo->prepare($sqlUpdate);
        $stmtUpdate->execute([
            ':cantidad' => $cantidad,
            ':es_urgente' => $es_urgente,
            ':existencia' => $existencia,
            ':pres_actual' => $pres_actual,
            ':pres_solicitada' => $pres_solicitada,
            ':detalle_id' => $detalle_id
        ]);
        
        // Si se envió usuario_id, actualizar el usuario del pedido principal
        // (Opcional: esto cambia el dueño de todo el pedido, tal vez no sea lo deseado si hay múltiples productos
        // pero en este modelo simple asumimos que el último que edita "toma" el pedido o se mantiene el original)
        if ($usuario_id) {
            $sqlUpdateUser = "UPDATE pedidos SET usuario_id = :usuario_id WHERE id = :pedido_id";
            $stmtUpdateUser = $pdo->prepare($sqlUpdateUser);
            $stmtUpdateUser->execute([
                ':usuario_id' => $usuario_id,
                ':pedido_id' => $detalle['pedido_id']
            ]);
        }
        
        $pdo->commit();
        
        sendJSON([
            'success' => true, 
            'message' => 'Producto actualizado correctamente',
            'detalle_id' => $detalle_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (PDOException $e) {
    error_log("Error al actualizar detalle: " . $e->getMessage());
    sendJSON(['success' => false, 'error' => 'Error de base de datos'], 500);
} catch (Exception $e) {
    error_log("Error general: " . $e->getMessage());
    sendJSON(['success' => false, 'error' => 'Error del servidor'], 500);
}
