<?php
require 'db.php';

try {
    // --- 1. Obtener la lista de todas las sucursales ---
    $sucursales_stmt = $pdo->query("SELECT nombre FROM sucursales ORDER BY nombre");
    $sucursales_list = $sucursales_stmt->fetchAll(PDO::FETCH_COLUMN);

    // --- 2. Obtener los datos (como en la API) ---
    $fecha_inicio = $_GET['fecha_inicio'] ?? '1900-01-01';
    $fecha_fin = $_GET['fecha_fin'] ?? '2999-12-31';
    if (empty($fecha_inicio)) $fecha_inicio = '1900-01-01';
    if (empty($fecha_fin)) $fecha_fin = '2999-12-31';

    $sql = "
        SELECT
            l.id AS lote_id, s.nombre AS nombre_sucursal, p.CodigoInterno,
            p.NombreComercial, p.laboratorio, l.codigo_lote, l.sefactura,
            DATE(l.fechacreacion) AS fechacreacion, i.stock_unidad_base,
            i.presentacion, p.CostoUnidad, p.CostoUnidadSinIva
        FROM inventario AS i
        JOIN lote AS l ON i.lote_id = l.id
        JOIN productos AS p ON l.producto_id = p.id
        JOIN sucursales AS s ON i.sucursal_id = s.id
        WHERE i.stock_unidad_base > 0 AND DATE(l.fechacreacion) BETWEEN ? AND ?
        ORDER BY p.NombreComercial, l.codigo_lote
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$fecha_inicio, $fecha_fin]);
    $resultados = $stmt->fetchAll();

    // --- 3. Pivotar los datos en PHP (misma lógica que la API) ---
    $pivoted_data = [];
    foreach ($resultados as $row) {
        $lote_id = $row['lote_id'];
        if (!isset($pivoted_data[$lote_id])) {
            $pivoted_data[$lote_id] = [
                'CodigoInterno' => $row['CodigoInterno'], 'NombreComercial' => $row['NombreComercial'],
                'laboratorio' => $row['laboratorio'], 'codigo_lote' => $row['codigo_lote'],
                'sefactura_texto' => ($row['sefactura'] == 1) ? 'Sí' : 'No',
                'fechacreacion' => $row['fechacreacion'], 'presentacion' => $row['presentacion'],
                'CostoUnidad' => $row['CostoUnidad'], 'CostoUnidadSinIva' => $row['CostoUnidadSinIva'],
                'StockTotal' => 0
            ];
            foreach ($sucursales_list as $sucursal_nombre) {
                $pivoted_data[$lote_id][$sucursal_nombre] = 0;
            }
        }
        $pivoted_data[$lote_id][$row['nombre_sucursal']] = (int)$row['stock_unidad_base'];
        $pivoted_data[$lote_id]['StockTotal'] += (int)$row['stock_unidad_base'];
    }

    // --- 4. Generar el CSV ---
    $filename = "reporte_inventario_pivotado_" . date('Y-m-d') . ".csv";
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF)); // BOM UTF-8

    // --- Escribir Cabeceras Dinámicas ---
    $static_headers = [
        'Codigo Interno', 'Nombre Comercial', 'Laboratorio', 'Lote', 
        'Se Factura?', 'Fecha Creacion', 'Presentacion', 
        'Costo Unidad', 'Costo Unidad S/IVA'
    ];
    
    // Combinar cabeceras estáticas + nombres de sucursales + Total
    $all_headers = array_merge($static_headers, $sucursales_list, ['Stock Total']);
    fputcsv($output, $all_headers, ';');

    // --- Escribir Filas de Datos ---
    foreach ($pivoted_data as $row) {
        $csv_row = [
            $row['CodigoInterno'],
            $row['NombreComercial'],
            $row['laboratorio'],
            $row['codigo_lote'],
            $row['sefactura_texto'],
            $row['fechacreacion'],
            $row['presentacion'],
            $row['CostoUnidad'],
            $row['CostoUnidadSinIva']
        ];
        
        // Añadir el stock de cada sucursal (en el orden correcto)
        foreach ($sucursales_list as $sucursal_nombre) {
            $csv_row[] = $row[$sucursal_nombre];
        }
        
        // Añadir el total
        $csv_row[] = $row['StockTotal'];
        
        fputcsv($output, $csv_row, ';');
    }

    fclose($output);

} catch (\PDOException $e) {
    // Si algo falla, se descarga un CSV con el error
    header('Content-Type: text/plain');
    echo "Error al generar el reporte: " . $e->getMessage();
}
exit();
?>