<?php
/**
 * API: Export Admin Orders to Excel
 * Exporta pedidos con soporte para múltiples sucursales en columnas separadas
 */

header('Access-Control-Allow-Origin: *');

require_once __DIR__ . '/../config/database.php';

try {
    $db = getDBConnection();
    
    // Get query parameters
    $sucursal_id = isset($_GET['sucursal_id']) ? (int)$_GET['sucursal_id'] : 0;
    $laboratorio = isset($_GET['laboratorio']) ? trim($_GET['laboratorio']) : '';
    $categoria_id = isset($_GET['categoria_id']) ? (int)$_GET['categoria_id'] : 0;
    $fecha_inicio = isset($_GET['fecha_inicio']) ? trim($_GET['fecha_inicio']) : '';
    $fecha_fin = isset($_GET['fecha_fin']) ? trim($_GET['fecha_fin']) : '';
    $es_urgente = isset($_GET['es_urgente']) ? (int)$_GET['es_urgente'] : 0;
    
    // Build WHERE clause
    $whereConditions = [];
    $params = [];
    
    if ($sucursal_id > 0) {
        $whereConditions[] = 'p.sucursal_id = ?';
        $params[] = $sucursal_id;
    }
    
    if (!empty($laboratorio)) {
        $whereConditions[] = 'prod.laboratorio LIKE ?';
        $params[] = "%{$laboratorio}%";
    }
    
    if ($categoria_id > 0) {
        $whereConditions[] = 'prod.id_categoria = ?';
        $params[] = $categoria_id;
    }
    
    if (!empty($fecha_inicio)) {
        $whereConditions[] = 'DATE(p.fecha_pedido) >= ?';
        $params[] = $fecha_inicio;
    }
    
    if (!empty($fecha_fin)) {
        $whereConditions[] = 'DATE(p.fecha_pedido) <= ?';
        $params[] = $fecha_fin;
    }
    
    if ($es_urgente > 0) {
        $whereConditions[] = 'dp.es_urgente = 1';
    }
    
    $whereClause = !empty($whereConditions) ? ' WHERE ' . implode(' AND ', $whereConditions) : '';
    
    // Determine filename
    $fecha_actual = date('Y-m-d');
    $filename = 'Pedidos_Admin_' . $fecha_actual;
    
    // Configure headers for Excel download
    header('Content-Type: application/vnd.ms-excel; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '.xls"');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    // Start HTML table for Excel
    echo '<!DOCTYPE html>';
    echo '<html>';
    echo '<head>';
    echo '<meta charset="UTF-8">';
    echo '<style>';
    echo 'table { border-collapse: collapse; width: 100%; }';
    echo 'th { background-color: #4472C4; color: white; font-weight: bold; padding: 8px; border: 1px solid #000; text-align: center; }';
    echo 'td { padding: 8px; border: 1px solid #000; }';
    echo 'td.number { text-align: center; }';
    echo 'h2 { color: #2E5090; font-family: Arial, sans-serif; }';
    echo '.urgent { background-color: #FFE6E6; }';
    echo '</style>';
    echo '</head>';
    echo '<body>';
    
    // Title
    echo '<h2>PEDIDOS CONSOLIDADOS - ' . date('d/m/Y') . '</h2>';
    
    if ($sucursal_id == 0) {
        // CONSOLIDATED VIEW - Multiple branches in columns
        
        // Get all branches
        $branchesStmt = $db->query("SELECT id, nombre FROM sucursales ORDER BY nombre");
        $branches = $branchesStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get aggregated data
        $sql = "SELECT 
                    dp.producto_id,
                    dp.codigo_interno,
                    dp.nombre_comercial,
                    prod.laboratorio,
                    c.nombre as categoria,
                    dp.presentacion_solicitada,
                    MAX(dp.es_urgente) as es_urgente,
                    p.sucursal_id,
                    SUM(dp.cantidad_solicitada) as cantidad_total
                FROM detalle_pedidos dp
                INNER JOIN pedidos p ON dp.pedido_id = p.id
                INNER JOIN productos prod ON dp.producto_id = prod.id
                LEFT JOIN categorias c ON prod.id_categoria = c.id" . 
                $whereClause .
                " GROUP BY dp.producto_id, p.sucursal_id, dp.presentacion_solicitada
                ORDER BY dp.nombre_comercial ASC";
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Reorganize data by product
        $products = [];
        foreach ($results as $row) {
            $productKey = $row['producto_id'] . '_' . $row['presentacion_solicitada'];
            
            if (!isset($products[$productKey])) {
                $products[$productKey] = [
                    'codigo_interno' => $row['codigo_interno'],
                    'nombre_comercial' => $row['nombre_comercial'],
                    'laboratorio' => $row['laboratorio'],
                    'categoria' => $row['categoria'] ?? 'Sin categoría',
                    'presentacion' => $row['presentacion_solicitada'],
                    'es_urgente' => $row['es_urgente'],
                    'sucursales' => []
                ];
            }
            
            $products[$productKey]['sucursales'][$row['sucursal_id']] = $row['cantidad_total'];
        }
        
        // Create table
        echo '<table>';
        
        // Headers
        echo '<tr>';
        echo '<th>Código</th>';
        echo '<th>Producto</th>';
        echo '<th>Laboratorio</th>';
        echo '<th>Categoría</th>';
        echo '<th>Presentación</th>';
        
        // Branch columns
        foreach ($branches as $branch) {
            echo '<th>' . htmlspecialchars($branch['nombre']) . '</th>';
        }
        
        echo '<th>Total</th>';
        echo '</tr>';
        
        // Data rows
        foreach ($products as $product) {
            $rowClass = $product['es_urgente'] ? ' class="urgent"' : '';
            echo '<tr' . $rowClass . '>';
            echo '<td>' . htmlspecialchars($product['codigo_interno'] ?? '') . '</td>';
            echo '<td>' . htmlspecialchars($product['nombre_comercial']) . '</td>';
            echo '<td>' . htmlspecialchars($product['laboratorio'] ?? '') . '</td>';
            echo '<td>' . htmlspecialchars($product['categoria']) . '</td>';
            echo '<td>' . htmlspecialchars($product['presentacion']) . '</td>';
            
            $total = 0;
            // Branch quantities
            foreach ($branches as $branch) {
                $cantidad = $product['sucursales'][$branch['id']] ?? 0;
                $total += $cantidad;
                echo '<td class="number">' . ($cantidad > 0 ? $cantidad : '-') . '</td>';
            }
            
            echo '<td class="number"><strong>' . $total . '</strong></td>';
            echo '</tr>';
        }
        
        echo '</table>';
        
    } else {
        // SINGLE BRANCH VIEW - Standard format
        
        // Get branch name
        $branchStmt = $db->prepare("SELECT nombre FROM sucursales WHERE id = ?");
        $branchStmt->execute([$sucursal_id]);
        $branchName = $branchStmt->fetchColumn();
        
        echo '<h3>Sucursal: ' . htmlspecialchars($branchName) . '</h3>';
        
        $sql = "SELECT 
                    dp.codigo_interno,
                    dp.nombre_comercial,
                    prod.laboratorio,
                    c.nombre as categoria,
                    dp.cantidad_solicitada,
                    dp.presentacion_solicitada,
                    dp.es_urgente
                FROM detalle_pedidos dp
                INNER JOIN pedidos p ON dp.pedido_id = p.id
                INNER JOIN productos prod ON dp.producto_id = prod.id
                LEFT JOIN categorias c ON prod.id_categoria = c.id" . 
                $whereClause .
                " ORDER BY dp.nombre_comercial ASC";
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Create table
        echo '<table>';
        
        // Headers
        echo '<tr>';
        echo '<th>Código</th>';
        echo '<th>Producto</th>';
        echo '<th>Laboratorio</th>';
        echo '<th>Categoría</th>';
        echo '<th>Cantidad</th>';
        echo '<th>Presentación</th>';
        echo '</tr>';
        
        // Data rows
        foreach ($products as $product) {
            $rowClass = $product['es_urgente'] ? ' class="urgent"' : '';
            echo '<tr' . $rowClass . '>';
            echo '<td>' . htmlspecialchars($product['codigo_interno'] ?? '') . '</td>';
            echo '<td>' . htmlspecialchars($product['nombre_comercial']) . '</td>';
            echo '<td>' . htmlspecialchars($product['laboratorio'] ?? '') . '</td>';
            echo '<td>' . htmlspecialchars($product['categoria'] ?? 'Sin categoría') . '</td>';
            echo '<td class="number">' . htmlspecialchars($product['cantidad_solicitada']) . '</td>';
            echo '<td>' . htmlspecialchars($product['presentacion_solicitada']) . '</td>';
            echo '</tr>';
        }
        
        echo '</table>';
    }
    
    echo '</body>';
    echo '</html>';
    
    exit;
    
} catch (PDOException $e) {
    error_log("Error al exportar pedidos admin: " . $e->getMessage());
    http_response_code(500);
    echo "Error al generar el archivo";
}
?>
