<?php
/**
 * API: Get Admin Orders
 * Obtiene pedidos para vista de administrador
 * Soporta vista de una sucursal específica o todas consolidadas
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

require_once __DIR__ . '/../config/database.php';

try {
    $db = getDBConnection();
    
    // Get query parameters
    $sucursal_id = isset($_GET['sucursal_id']) ? (int)$_GET['sucursal_id'] : 0;
    $laboratorio = isset($_GET['laboratorio']) ? trim($_GET['laboratorio']) : '';
    $categoria_id = isset($_GET['categoria_id']) ? (int)$_GET['categoria_id'] : 0;
    $fecha_inicio = isset($_GET['fecha_inicio']) ? trim($_GET['fecha_inicio']) : '';
    $fecha_fin = isset($_GET['fecha_fin']) ? trim($_GET['fecha_fin']) : '';
    $es_urgente = isset($_GET['es_urgente']) ? (int)$_GET['es_urgente'] : 0;
    
    // Build WHERE clause
    $whereConditions = [];
    $params = [];
    
    // Filter by branch if specified
    if ($sucursal_id > 0) {
        $whereConditions[] = 'p.sucursal_id = ?';
        $params[] = $sucursal_id;
    }
    
    // Filter by laboratory
    if (!empty($laboratorio)) {
        $whereConditions[] = 'prod.laboratorio LIKE ?';
        $params[] = "%{$laboratorio}%";
    }
    
    // Filter by category
    if ($categoria_id > 0) {
        $whereConditions[] = 'prod.id_categoria = ?';
        $params[] = $categoria_id;
    }
    
    // Filter by date range
    if (!empty($fecha_inicio)) {
        $whereConditions[] = 'DATE(p.fecha_pedido) >= ?';
        $params[] = $fecha_inicio;
    }
    
    if (!empty($fecha_fin)) {
        $whereConditions[] = 'DATE(p.fecha_pedido) <= ?';
        $params[] = $fecha_fin;
    }
    
    // Filter by urgent
    if ($es_urgente > 0) {
        $whereConditions[] = 'dp.es_urgente = 1';
    }
    
    $whereClause = !empty($whereConditions) ? ' WHERE ' . implode(' AND ', $whereConditions) : '';
    
    // If viewing all branches, group by product and show quantities per branch
    if ($sucursal_id == 0) {
        // Get all branches for column headers
        $branchesStmt = $db->query("SELECT id, nombre FROM sucursales ORDER BY nombre");
        $branches = $branchesStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get aggregated data grouped by product
        $sql = "SELECT 
                    dp.producto_id,
                    dp.codigo_interno,
                    dp.codigo_barra,
                    dp.nombre_comercial,
                    prod.laboratorio,
                    c.nombre as categoria,
                    dp.presentacion_solicitada,
                    MAX(dp.es_urgente) as es_urgente,
                    p.sucursal_id,
                    s.nombre as sucursal_nombre,
                    SUM(dp.cantidad_solicitada) as cantidad_total
                FROM detalle_pedidos dp
                INNER JOIN pedidos p ON dp.pedido_id = p.id
                INNER JOIN productos prod ON dp.producto_id = prod.id
                INNER JOIN sucursales s ON p.sucursal_id = s.id
                LEFT JOIN categorias c ON prod.id_categoria = c.id" . 
                $whereClause .
                " GROUP BY dp.producto_id, p.sucursal_id, dp.presentacion_solicitada
                ORDER BY dp.nombre_comercial ASC, s.nombre ASC";
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Reorganize data by product
        $products = [];
        foreach ($results as $row) {
            $productKey = $row['producto_id'] . '_' . $row['presentacion_solicitada'];
            
            if (!isset($products[$productKey])) {
                $products[$productKey] = [
                    'producto_id' => $row['producto_id'],
                    'codigo_interno' => $row['codigo_interno'],
                    'codigo_barra' => $row['codigo_barra'],
                    'nombre_comercial' => $row['nombre_comercial'],
                    'laboratorio' => $row['laboratorio'],
                    'categoria' => $row['categoria'],
                    'presentacion_solicitada' => $row['presentacion_solicitada'],
                    'es_urgente' => $row['es_urgente'],
                    'sucursales' => []
                ];
            }
            
            $products[$productKey]['sucursales'][$row['sucursal_id']] = [
                'nombre' => $row['sucursal_nombre'],
                'cantidad' => $row['cantidad_total']
            ];
        }
        
        echo json_encode([
            'success' => true,
            'view_mode' => 'consolidated',
            'branches' => $branches,
            'data' => array_values($products)
        ]);
        
    } else {
        // Single branch view - similar to existing get_ordered_products.php
        $sql = "SELECT 
                    dp.*,
                    p.id as pedido_id,
                    p.fecha_pedido,
                    p.estado as pedido_estado,
                    s.nombre as sucursal_nombre,
                    u.nombre as usuario_nombre,
                    prod.laboratorio,
                    c.nombre as categoria
                FROM detalle_pedidos dp
                INNER JOIN pedidos p ON dp.pedido_id = p.id
                INNER JOIN sucursales s ON p.sucursal_id = s.id
                INNER JOIN usuarios u ON p.usuario_id = u.id
                INNER JOIN productos prod ON dp.producto_id = prod.id
                LEFT JOIN categorias c ON prod.id_categoria = c.id" . 
                $whereClause .
                " ORDER BY p.fecha_pedido DESC, dp.nombre_comercial ASC";
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        $products = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'view_mode' => 'single',
            'data' => $products
        ]);
    }
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error al obtener pedidos: ' . $e->getMessage()
    ]);
}
?>
