<?php
header('Content-Type: application/json');
require_once '../../db_connection.php';

// Leer el cuerpo de la solicitud
$data = json_decode(file_get_contents('php://input'), true);

// --- Validación de Datos ---
if (!$data || !isset($data['sucursal_origen_id'], $data['sucursal_destino_id'], $data['productos'])) {
    echo json_encode(['success' => false, 'message' => 'Datos incompletos.']);
    exit;
}

$origen_id = (int)$data['sucursal_origen_id'];
$destino_id = (int)$data['sucursal_destino_id'];
$observaciones = isset($data['observaciones']) ? trim($data['observaciones']) : null;
$productos = $data['productos'];
$usuario_id = 1; // Temporal: Debería obtenerse de la sesión del usuario

if ($origen_id === $destino_id) {
    echo json_encode(['success' => false, 'message' => 'La sucursal de origen y destino no pueden ser la misma.']);
    exit;
}

if (empty($productos)) {
    echo json_encode(['success' => false, 'message' => 'No se han añadido productos al traslado.']);
    exit;
}

// --- Inicio de la Transacción ---
$conn->begin_transaction();

try {
    // 1. Crear el registro maestro del traslado
    $stmt_traslado = $conn->prepare(
        "INSERT INTO traslados (sucursal_origen, sucursal_destino, usuario_id, estado, observaciones) VALUES (?, ?, ?, 'completado', ?)"
    );
    $stmt_traslado->bind_param("iiis", $origen_id, $destino_id, $usuario_id, $observaciones);
    $stmt_traslado->execute();
    $traslado_id = $stmt_traslado->insert_id;
    $stmt_traslado->close();

    if (!$traslado_id) {
        throw new Exception("No se pudo crear el registro de traslado.");
    }

    // 2. Procesar cada producto
    foreach ($productos as $p) {
        $lote_id = (int)$p['lote_id'];
        $producto_id = (int)$p['producto_id'];
        $cantidad = (int)$p['cantidad'];

        if ($cantidad <= 0) continue; // Omitir si la cantidad no es válida

        // Bloquear la fila en inventario para evitar concurrencia
        $stmt_lock = $conn->prepare("SELECT stock_unidad_base FROM inventario WHERE lote_id = ? AND sucursal_id = ? FOR UPDATE");
        $stmt_lock->bind_param("ii", $lote_id, $origen_id);
        $stmt_lock->execute();
        $result_stock = $stmt_lock->get_result();

        if ($result_stock->num_rows === 0) {
            throw new Exception("El producto/lote ID {$lote_id} no existe en la sucursal de origen.");
        }

        $current_stock = $result_stock->fetch_assoc()['stock_unidad_base'];
        $stmt_lock->close();

        if ($current_stock < $cantidad) {
            throw new Exception("Stock insuficiente para el lote ID {$lote_id}. Disponible: {$current_stock}, Solicitado: {$cantidad}");
        }

        // Restar stock del origen
        $stmt_origen = $conn->prepare("UPDATE inventario SET stock_unidad_base = stock_unidad_base - ? WHERE lote_id = ? AND sucursal_id = ?");
        $stmt_origen->bind_param("iii", $cantidad, $lote_id, $origen_id);
        $stmt_origen->execute();
        $stmt_origen->close();

        // Añadir stock al destino (o crear el registro si no existe)
        $stmt_destino = $conn->prepare(
            "INSERT INTO inventario (lote_id, sucursal_id, stock_unidad_base, presentacion) 
             VALUES (?, ?, ?, (SELECT id_presentacion FROM productos WHERE id = ?)) 
             ON DUPLICATE KEY UPDATE stock_unidad_base = stock_unidad_base + ?"
        );
        // Nota: Asumimos que la presentación se puede obtener del producto. Ajustar si es necesario.
        $stmt_destino->bind_param("iiiii", $lote_id, $destino_id, $cantidad, $producto_id, $cantidad);
        $stmt_destino->execute();
        $stmt_destino->close();

        // Insertar en detalle_traslados
        // Simplificado a una sola cantidad. Ajustar si se necesita el detalle de caja/blister.
        $stmt_detalle = $conn->prepare("INSERT INTO detalle_traslados (traslado_id, producto_id, cantidad_unidad) VALUES (?, ?, ?)");
        $stmt_detalle->bind_param("iii", $traslado_id, $producto_id, $cantidad);
        $stmt_detalle->execute();
        $stmt_detalle->close();
    }

    // 3. Si todo fue bien, confirmar la transacción
    $conn->commit();
    echo json_encode(['success' => true, 'message' => 'Traslado completado con éxito.', 'traslado_id' => $traslado_id]);

} catch (Exception $e) {
    // 4. Si algo falló, revertir la transacción
    $conn->rollback();
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

$conn->close();
?>
