document.addEventListener('DOMContentLoaded', function() {
    const sucursalOrigenSelect = document.getElementById('sucursal-origen');
    const sucursalDestinoSelect = document.getElementById('sucursal-destino');
    const productoSearchInput = document.getElementById('producto-search');
    const suggestionsDiv = document.getElementById('producto-suggestions');
    const transferTableBody = document.querySelector('#transfer-products-table tbody');
    const transferForm = document.getElementById('transfer-form');

    let transferProducts = []; // Almacenará los productos a transferir

    // Cargar sucursales
    function loadSucursales() {
        fetch('../api/get_sucursales.php')
            .then(response => response.json())
            .then(data => {
                sucursalOrigenSelect.innerHTML = '<option value="">Seleccione origen</option>';
                sucursalDestinoSelect.innerHTML = '<option value="">Seleccione destino</option>';
                data.forEach(sucursal => {
                    sucursalOrigenSelect.innerHTML += `<option value="${sucursal.id}">${sucursal.nombre}</option>`;
                    sucursalDestinoSelect.innerHTML += `<option value="${sucursal.id}">${sucursal.nombre}</option>`;
                });
            });
    }

    // Validar que origen y destino no sean iguales
    function validateSucursales() {
        if (sucursalOrigenSelect.value && sucursalOrigenSelect.value === sucursalDestinoSelect.value) {
            alert('La sucursal de origen y destino no pueden ser la misma.');
            sucursalDestinoSelect.value = '';
        }
    }

    sucursalOrigenSelect.addEventListener('change', validateSucursales);
    sucursalDestinoSelect.addEventListener('change', validateSucursales);

    // Buscar productos
    productoSearchInput.addEventListener('input', function() {
        const searchTerm = this.value;
        const sucursalOrigenId = sucursalOrigenSelect.value;

        if (searchTerm.length < 2 || !sucursalOrigenId) {
            suggestionsDiv.innerHTML = '';
            return;
        }

        // Usaremos get_products.php, pero necesitamos adaptarlo o crear uno nuevo 
        // que filtre por sucursal y stock.
        // Por ahora, simulamos la búsqueda.
        fetch(`../api/get_products_for_transfer.php?term=${searchTerm}&sucursal_id=${sucursalOrigenId}`)
            .then(response => response.json())
            .then(data => {
                suggestionsDiv.innerHTML = '';
                if (data.length > 0) {
                    data.forEach(producto => {
                        const div = document.createElement('div');
                        div.innerHTML = `${producto.NombreComercial} (Lote: ${producto.codigo_lote}) - Stock: ${producto.stock_unidad_base}`;
                        div.classList.add('suggestion-item');
                        div.addEventListener('click', () => addProductToTransfer(producto));
                        suggestionsDiv.appendChild(div);
                    });
                }
            });
    });

    // Añadir producto a la tabla de traslados
    function addProductToTransfer(producto) {
        productoSearchInput.value = '';
        suggestionsDiv.innerHTML = '';

        if (transferProducts.find(p => p.lote_id === producto.lote_id)) {
            alert('Este lote de producto ya ha sido añadido.');
            return;
        }

        transferProducts.push(producto);
        renderTransferTable();
    }

    // Renderizar la tabla de productos a transferir
    function renderTransferTable() {
        transferTableBody.innerHTML = '';
        transferProducts.forEach((producto, index) => {
            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${producto.NombreComercial}</td>
                <td>${producto.codigo_lote}</td>
                <td>${producto.stock_unidad_base}</td>
                <td><input type="number" class="transfer-quantity" data-index="${index}" min="1" max="${producto.stock_unidad_base}" required></td>
                <td><button type="button" class="remove-btn" data-index="${index}">Quitar</button></td>
            `;
            transferTableBody.appendChild(row);
        });
    }

    // Quitar producto de la tabla
    transferTableBody.addEventListener('click', function(e) {
        if (e.target.classList.contains('remove-btn')) {
            const index = e.target.getAttribute('data-index');
            transferProducts.splice(index, 1);
            renderTransferTable();
        }
    });

    // Enviar formulario
    transferForm.addEventListener('submit', function(e) {
        e.preventDefault();

        const items = [];
        const quantityInputs = document.querySelectorAll('.transfer-quantity');
        let valid = true;

        quantityInputs.forEach(input => {
            const index = input.getAttribute('data-index');
            const quantity = parseInt(input.value, 10);
            const product = transferProducts[index];

            if (quantity > 0 && quantity <= product.stock_unidad_base) {
                items.push({
                    lote_id: product.lote_id,
                    producto_id: product.producto_id, // Necesitamos el producto_id
                    cantidad: quantity
                });
            } else {
                valid = false;
            }
        });

        if (!valid || items.length === 0) {
            alert('Por favor, verifique las cantidades a trasladar.');
            return;
        }

        const transferData = {
            sucursal_origen_id: sucursalOrigenSelect.value,
            sucursal_destino_id: sucursalDestinoSelect.value,
            observaciones: document.getElementById('observaciones').value,
            productos: items
        };

        fetch('../api/create_transfer.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(transferData)
        })
        .then(response => response.json())
        .then(result => {
            if (result.success) {
                alert('Traslado realizado con éxito.');
                transferForm.reset();
                transferProducts = [];
                renderTransferTable();
            } else {
                alert('Error al realizar el traslado: ' + result.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Ocurrió un error de conexión.');
        });
    });

    // Carga inicial
    loadSucursales();
});
